# Réalisé par Denis Duplan pour Stash of Code (https://www.stashofcode.fr) en novembre 2021.

# Cette oeuvre est mise à disposition selon les termes de la Licence Creative Commons Attribution - Pas d’Utilisation Commerciale 4.0 International

# Etape 2

# Test de désérialisation / sérialisation des données brutes de blocs de la blockchain BTC à partir de l'extraction d'une série de blocs consécutifs produite par downloadBlocks.py sous forme d'un fichier JSON (les données brutes d'un bloc se trouvent dans l'entrée "hex" du bloc). On génére une désérialisation lisible de tous les blocs dans un unique fichier au format TXT ainsi qu'une sérialisation de chaque bloc dans un fichier au format BIN.

# Noter que la validation de la désérialisation du hash de la transaction de l'input d'une transaction échoue tout le temps, car dans le JSON, ce hash est remplacé par des informations dont j'ignore encore l'origine (cf. "prev_out" dans le JSON), à moins qu'il ne faille chercher ce hash désormais dans les données du witness de l'input ?

# Noter qu'a priori les valeurs des champs ainsi récupérées sont en big endian, si bien qu'aucune inversion n'est nécessaire pour les transformer en valeurs qu'il est possible de manipuler en Python. Toutefois, la chose curieuse, c'est qu'une fois le hash calculé sur le header sérialisé en little endian, il faut inverser l'endianess du hash pour retrouver le hash en big endian (ie : un hash commençant par des 0) ? C'est du moins ainsi que cela pervient à fonctionner, et cela recoupe le code PHP fourni sur :

# https://en.bitcoin.it/wiki/Block_hashing_algorithm

# Noter que les données relatives aux witnesses (le flag qui en mentionne la présence, et les witnesses eux-mêmes) sont exclus des données à partir desquelles le hash d'une transaction est calculé.

# Pour ce qui concerne le script, la manière de le décoder est présentée ici :

# https://en.bitcoin.it/wiki/Script

# Attention! tout comme il est spécifié "The Bitcoin protocol is specified by the behavior of the reference client, not by this page", il est spécifié que "De facto, Bitcoin script is defined by the code run by the network to check the validity of blocks". Bref, la doc, c'est le code, ici encore, on dirait... La référence ultime serait donc le code de Bitcoin Core, présenté comme "a direct descendant of the original Bitcoin software client released by Satoshi Nakamoto after he published the famous Bitcoin whitepaper", qui contient le code d'un full-node et d'un wallet :

# https://bitcoincore.org/en/about/
# https://github.com/bitcoin/bitcoin

# Mais pour ce qui concerne l'exercice présent, qui consiste à manipuler les premiers blocs, le mieux est de se reporter au code de la premire version de Bitcoin, la version 0.1.0, disponible ici :

# https://satoshi.nakamotoinstitute.org/code/

# En effet, Bitcoin a beaucoup évolué depuis, et lire ce code est non seulement beaucoup simple car il est plus léger, n'étant pas encombré par des fonctionnalités qui ont été ajoutées, mais permet de plus d'éviter de commettre des anachronismes en faisant le tri entre les fonctionnalités d'origine et ces dernières.


import json
import datetime
import calendar
import hashlib

def bytesToString (data):
	return ''.join ([f'{b:02x}' for b in data])
def bytesToASCII (data):
	return ''.join ([chr (b) if b in range (0x20, 0x7f) else '.' for b in data])
def stringToBytes (data):
	return bytearray ([int (data[i:i + 2], 16) for i in range (0, len (data), 2)])

# https://en.bitcoin.it/wiki/Protocol_documentation#Variable_length_integer
def variantToBytes (n):
	data = bytearray ()
	if n < 0xfd:
		size = 1
	elif n < 0xffff:
		size = 2
		data.append (0xfd)
	elif n < 0xffffffff:
		size = 4
		data.append (0xfe)
	else:
		size = 8
		data.append (0xff)
	data.extend (n.to_bytes (size, byteorder='little'))
	return data, len (data)

def variantFromBytes (data):
	if data[0] == 0xff:
		ofs = 1
		size = 8
	elif data[0] == 0xfe:
		ofs = 1
		size = 4
	elif data[0] == 0xfd:
		ofs = 1
		size = 2
	else:
		ofs = 0
		size = 1
	return int.from_bytes (data[ofs:ofs + size], byteorder='little'), ofs + size

# https://en.bitcoin.it/wiki/Script
# https://github.com/bitcoin/bitcoin/blob/master/src/script/script.h
# La pile contient des "bytes vectors", de séquences de 520 octets au plus. Quand une séquence est un nombre, c'est un entier sur 4 octets en little endian, dont le bit le plus significatif donne le signe (ex : 0x81 = -1). Noter que l'addition et la soustraction peuvent produire un entier sur 5 octets. Quand c'est un booléen, c'est la valeur False si au moins un des octets n'est pas nul.
def decodeScript (script, prefix=''):
	opcodes = {
		0x61: 'OP_NOP',
		0x63: 'OP_IF',
		0x64: 'OP_NOTIF',
		0x67: 'OP_ELSE',
		0x68: 'OP_ENDIF',
		0x69: 'OP_VERIFY',
		0x6a: 'OP_RETURN',
		0x6b: 'OP_TOALTSTACK',
		0x6c: 'OP_FROMALTSTACK',
		0x73: 'OP_IFDUP',
		0x74: 'OP_DEPTH',
		0x75: 'OP_DROP',
		0x76: 'OP_DUP',
		0x77: 'OP_NIP',
		0x78: 'OP_OVER',
		0x79: 'OP_PICK',
		0x7a: 'OP_ROLL',
		0x7b: 'OP_ROT',
		0x7c: 'OP_SWAP',
		0x7d: 'OP_TUCK',
		0x6d: 'OP_2DROP',
		0x6e: 'OP_2DUP',
		0x6f: 'OP_3DUP',
		0x70: 'OP_2OVER',
		0x71: 'OP_2ROT',
		0x72: 'OP_2SWAP',
		0x7e: 'OP_CAT',
		0x7f: 'OP_SUBSTR',
		0x80: 'OP_LEFT',
		0x81: 'OP_RIGHT',
		0x82: 'OP_SIZE',
		0x83: 'OP_INVERT',
		0x84: 'OP_AND',
		0x85: 'OP_OR',
		0x86: 'OP_XOR',
		0x87: 'OP_EQUAL',
		0x88: 'OP_EQUALVERIFY',
		0x8b: 'OP_1ADD',
		0x8c: 'OP_1SUB',
		0x8d: 'OP_2MUL',
		0x8e: 'OP_2DIV',
		0x8f: 'OP_NEGATE',
		0x90: 'OP_ABS',
		0x91: 'OP_NOT',
		0x92: 'OP_0NOTEQUAL',
		0x93: 'OP_ADD',
		0x94: 'OP_SUB',
		0x95: 'OP_MUL',
		0x96: 'OP_DIV',
		0x97: 'OP_MOD',
		0x98: 'OP_LSHIFT',
		0x99: 'OP_RSHIFT',
		0x9a: 'OP_BOOLAND',
		0x9b: 'OP_BOOLOR',
		0x9c: 'OP_NUMEQUAL',
		0x9d: 'OP_NUMEQUALVERIFY',
		0x9e: 'OP_NUMNOTEQUAL',
		0x9f: 'OP_LESSTHAN',
		0xa0: 'OP_GREATERTHAN',
		0xa1: 'OP_LESSTHANOREQUAL',
		0xa2: 'OP_GREATERTHANOREQUAL',
		0xa3: 'OP_MIN',
		0xa4: 'OP_MAX',
		0xa5: 'OP_WITHIN',
		0xa6: 'OP_RIPEMD160',
		0xa7: 'OP_SHA1',
		0xa8: 'OP_SHA256',
		0xa9: 'OP_HASH160',
		0xaa: 'OP_HASH256',
		0xab: 'OP_CODESEPARATOR',
		0xac: 'OP_CHECKSIG',
		0xad: 'OP_CHECKSIGVERIFY',
		0xae: 'OP_CHECKMULTISIG',
		0xaf: 'OP_CHECKMULTISIGVERIFY',
		0xb1: 'OP_CHECKLOCKTIMEVERIFY (ex-OP_NOP2)',
		0xb2: 'OP_CHECKSEQUENCEVERIFY (ex-OP_NOP3)',
		0xfd: 'OP_PUBKEYHASH',
		0xfe: 'OP_PUBKEY',
		0xff: 'OP_INVALIDOPCODE',
		0x50: 'OP_RESERVED',
		0x62: 'OP_VER',
		0x65: 'OP_VERIF',
		0x66: 'OP_VERNOTIF',
		0x89: 'OP_RESERVED1',
		0x8a: 'OP_RESERVED2',
		0xb0: 'OP_NOP1',
		0xb3: 'OP_NOP4',
		0xb4: 'OP_NOP5',
		0xb5: 'OP_NOP6',
		0xb6: 'OP_NOP7',
		0xb7: 'OP_NOP8',
		0xb8: 'OP_NOP9',
		0xb9: 'OP_NOP10'
	}
	s = ''
	i = 0
	while i < len (script):
		b = script[i]
		if b == 0:
			s += f'{prefix}PUSH empty array\n'
			i += 1
		elif b >= 1 and b <= 75:
			i += 1
			s += f'{prefix}PUSH {b} bytes: {bytesToString (script[i:i + b])}\n'
			s += prefix + len (f'PUSH {b} bytes: ') * ' ' + f'{bytesToASCII (script[i:i + b])}\n'
			i += b
		elif b >= 76 and b <= 78:
			b = int.from_bytes (script[i:i + 76 - b + 1], byteorder='little')
			i += 76 - b + 1
			s += f'{prefix}PUSH {b} bytes: {bytesToString (script[i:i + b])}\n'
			s += prefix + len (f'PUSH {b} bytes: ') * ' ' + f'{bytesToASCII (script[i:i + b])}\n'
			i += b
		elif b == 79:
			s += f'{prefix}PUSH -1\n'
			i += 1
		elif b == 81:
			s += f'{prefix}PUSH 1\n'
			i += 1
		elif b >= 82 and b <= 96:
			s += f'{prefix}PUSH {b - 82 + 2}\n'
			i += 1
		else:
			if b not in opcodes:
				s += f'{prefix}*UNKNOWN*: {bytesToString (script[i:])}\n'
				s += prefix + len (f'*UNKNOWN*: ') * ' ' + f'{bytesToASCII (script[i:])}\n'
				break
			else:
				s += f'{prefix}{opcodes[b]}\n'
				i += 1
	return s

# https://github.com/bitcoin/bitcoin/blob/master/src/primitives/transaction.h
# https://en.bitcoin.it/wiki/Protocol_documentation#tx
# https://en.bitcoin.it/wiki/Transactions
class Transaction:

	def validate (self, transaction):
		isValid = True
		if self.version != transaction['ver']:
			print (f'Bad version: {self.version} instead of {transaction["ver"]}')
			isValid = False
		if len (self.inputs) != transaction['vin_sz']:
			print (f'Bad  # of inputs: {len (self.inputs)} instead of {transaction["vin_sz"]}')
			isValid = False
		for i in range (len (self.inputs)):
			isValid = isValid and self.inputs[i].validate (transaction['inputs'][i])
			if transaction['inputs'][i]['witness'] != '':
				isValid = isValid and self.witnesses[i].validate (transaction['inputs'][i]['witness'])
		if len (self.outputs) != transaction['vout_sz']:
			print (f'Bad  # of outputs: {len (self.outputs)} instead of {transaction["vout_sz"]}')
			isValid = False
		for i in range (len (self.outputs)):
			isValid = isValid and self.outputs[i].validate (transaction['out'][i])
		if self.lockTime != transaction['lock_time']:
			print (f'Bad lock time: {self.lockTime} instead of {transaction["lock_time"]}')
			isValid = False
		return isValid

	def dump (self, prefix):
		separator = f'{prefix}\t{"-" * 80}\n'
		s = f'{prefix}Version: {self.version}\n'
		if self.lockTime == 0:
			lockTime = 'Not locked'
		elif self.lockTime < 500000000:
			lockTime = f'Block #{self.lockTime}'
		else:
			lockTime = datetime.datetime.utcfromtimestamp (self.lockTime)
		s += f'{prefix}Lock time: {lockTime}\n'
		s += f'{prefix}Inputs: {len (self.inputs)}\n'
		for i in range (len (self.inputs)):
			s += f'{separator}{prefix}\tInput #{i:03d}\n{separator}'
			s += self.inputs[i].dump (prefix + '\t')
		s += f'{prefix}Outputs: {len (self.outputs)}\n'
		for i in range (len (self.outputs)):
			s += f'{separator}{prefix}\tOutput #{i:03d}\n{separator}'
			s += self.outputs[i].dump (prefix + '\t')
		s += f'{prefix}Witnesses: {len (self.witnesses)}\n'
		for i in range (len (self.witnesses)):
			s += f'{separator}{prefix}\tWitness #{i:03d}\n{separator}'
			s += self.witnesses[i].dump (prefix + '\t')
		return s

	def serialize (self, withWitnesses=True):
		data = bytearray ()
		data.extend (self.version.to_bytes (4, byteorder='little'))
		if withWitnesses:
			if len (self.witnesses):
				data.append (0x00)
				data.append (0x01)
		data.extend (variantToBytes (len (self.inputs))[0])
		for txIn in self.inputs:
			data.extend (txIn.serialize ())
		data.extend (variantToBytes (len (self.outputs))[0])
		for txOut in self.outputs:
			data.extend (txOut.serialize ())
		if withWitnesses:
			if len (self.witnesses):
				for witness in self.witnesses:
					data.extend (witness.serialize ())
		data.extend (self.lockTime.to_bytes (4, byteorder='little'))
		return data

	def deserialize (self, data, progress=''):
		ofs = 0
		# Version (4 octets)
		self.version = int.from_bytes (data[ofs:ofs + 4], byteorder='little')
		ofs += 4
		# Présence de témoins (4 octets, si 0x0001)
		hasWitnesses = False
		if int.from_bytes (data[ofs:ofs + 2], byteorder='little') == 0x0100:
			hasWitnesses = True
			ofs += 2
		# Nombre d'entrées (1 à 9 octets)
		variant = variantFromBytes (data[ofs:])
		nbInputs = variant[0]
		ofs += variant[1]
		# Entrées
		self.inputs = []
		for i in range (nbInputs):
			print (f'\r{progress}Input: {i + 1} / {nbInputs}', end='')
			transactionInput = TransactionInput ()
			ofs += transactionInput.deserialize (data[ofs:])
			self.inputs.append (transactionInput)
		# Nombre de sorties (1 à 9 octets)
		variant = variantFromBytes (data[ofs:])
		nbOutputs = variant[0]
		ofs += variant[1]
		# Sorties
		self.outputs = []
		for i in range (nbOutputs):
			print (f'\r{progress}Output: {i + 1} / {nbOutputs}', end='')
			transactionOutput = TransactionOutput ()
			ofs += transactionOutput.deserialize (data[ofs:])
			self.outputs.append (transactionOutput)
		# Témoins
		self.witnesses = []
		if hasWitnesses:
			for i in range (nbInputs):
				print (f'\r{progress}Witness: {i + 1} / {nbInputs}', end='')
				witness = Witness ()
				ofs += witness.deserialize (data[ofs:])
				self.witnesses.append (witness)
		# Hauteur du bloc ou date & heure de finalisation du bloc (4 octets)
		self.lockTime = int.from_bytes (data[ofs:ofs + 4], byteorder='little')
		ofs += 4
		return ofs

	# https://github.com/bitcoin/bips/blob/master/bip-0141.mediawiki
	def hash (self):
		return bytearray (hashlib.sha256 (hashlib.sha256 (self.serialize (withWitnesses=False)).digest ()).digest ())

# https://en.bitcoin.it/wiki/Protocol_documentation#tx
# https://en.bitcoin.it/wiki/Transactions
class TransactionInput:

	def validate (self, transactionInput):
		isValid = True
		prev_out = transactionInput['prev_out']
		if prev_out is None:
			prev_out = bytearray (b'\x00' * 32)
		if self.transaction != prev_out:
			print (f'Bad previous transaction: {self.transaction} instead of {prev_out}')
			isValid = False
		if self.scriptSig != stringToBytes (transactionInput['script']):
			print (f'Bad ScriptSig: {self.scriptSig} instead of {transactionInput["script"]}')
			isValid = False
		if self.sequence != transactionInput['sequence']:
			print (f'Bad sequence: {self.sequence} instead of {transactionInput["sequence"]}')
			isValid = False
		return isValid

	def dump (self, prefix):
		s = f'{prefix}Previous transaction double SHA-256: {bytesToString (self.transaction)}\n'
		s += f'{prefix}Previous transaction output: {self.output}\n'
		s += f'{prefix}ScriptSig: {bytesToString (self.scriptSig)}\n'
		s += decodeScript (self.scriptSig, f'{prefix}\t')
		s += f'{prefix}Sequence: {self.sequence}\n'
		return s

	def serialize (self):
		data = bytearray ()
		hash = bytearray (self.transaction)
		hash.reverse ()
		data.extend (hash)
		data.extend (self.output.to_bytes (4, byteorder='little'))
		data.extend (variantToBytes (len (self.scriptSig))[0])
		data.extend (self.scriptSig)
		data.extend (self.sequence.to_bytes (4, byteorder='little'))
		return data

	def deserialize (self, data):
		ofs = 0
		# Double SHA-256 de la transaction précédente (32 octets)
		self.transaction = data[ofs:ofs + 32]
		self.transaction.reverse ()
		ofs += 32
		# Indice de la sortie de la transaction précédente (4 octets)
		self.output = int.from_bytes (data[ofs:ofs + 4], byteorder='little')
		ofs += 4
		# Taille du script (1 à 9 octets)
		variant = variantFromBytes (data[ofs:])
		scriptSigSize = variant[0]
		ofs += variant[1]
		# Script
		self.scriptSig = data[ofs:ofs + scriptSigSize]
		ofs += scriptSigSize
		# Séquence (4 octets)
		self.sequence = int.from_bytes (data[ofs:ofs + 4], byteorder='little')
		ofs += 4
		return ofs

# https://en.bitcoin.it/wiki/Protocol_documentation#tx
# https://en.bitcoin.it/wiki/Transactions
class TransactionOutput:

	def validate (self, transactionOutput):
		isValid = True
		if self.value != transactionOutput['value']:
			print (f'Bad value: {self.value} instead of {transactionOutput["prev_out"]}')
			isValid = False
		if self.scriptPubKey != stringToBytes (transactionOutput['script']):
			print (f'Bad ScriptPubKey: {self.scriptPubKey} instead of {transactionOutput["script"]}')
			isValid = False
		return isValid

	def dump (self, prefix):
		s = f'{prefix}Value: {self.value} Satoshis ({self.value / 10**8} BTC)\n'
		s += f'{prefix}ScriptPubKey: {bytesToString (self.scriptPubKey)}\n'
		s += decodeScript (self.scriptPubKey, f'{prefix}\t')
		return s

	def serialize (self):
		data = bytearray ()
		data.extend (self.value.to_bytes (8, byteorder='little'))
		data.extend (variantToBytes (len (self.scriptPubKey))[0])
		data.extend (self.scriptPubKey)
		return data

	def deserialize (self, data):
		ofs = 0
		# Valeur en Satoshis (ie : BTC / 10^8) (8 octets)
		self.value = int.from_bytes (data[ofs:ofs + 8], byteorder='little')
		ofs += 8
		# Taille du script (1 à 9 octets)
		variant = variantFromBytes (data[ofs:])
		scriptPubKeySize = variant[0]
		ofs += variant[1]
		# Script
		self.scriptPubKey = data[ofs:ofs + scriptPubKeySize]
		ofs += scriptPubKeySize
		return ofs

# https://en.bitcoin.it/wiki/Protocol_documentation#tx
# https://en.bitcoin.it/wiki/Segregated_Witness
class Witness:

	def validate (self, witness):
		# Dans le JSON, les données d'un witness sont brutes et ne peuvent donc être validées
		isValid = True
		data = self.serialize ()
		if data != stringToBytes (witness):
			print (f'Bad witness: {bytesToString (data)} instead of {witness}')
			isValid = False
		return isValid

	def dump (self, prefix):
		separator = f'{prefix}\t{"-" * 80}\n'
		s = f'{prefix}# of components: {len (self.components)}\n'
		s += f'{prefix}Components: {len (self.components)}\n'
		for i in range (len (self.components)):
			s += f'{separator}{prefix}\tComponent #{i:03d}\n{separator}'
			s += f'{prefix}\tSize: {len (self.components[i])}\n'
			s += f'{prefix}\tData: {bytesToString (self.components[i])}\n'
		return s

	def serialize (self):
		data = bytearray ()
		data.extend (variantToBytes (len (self.components))[0])
		for component in self.components:
			data.extend (variantToBytes (len (component))[0])
			data.extend (component)
		return data

	def deserialize (self, data):
		ofs = 0
		# Nombre de composants (1 à 9 octets)
		variant = variantFromBytes (data[ofs:])
		nbComponents = variant[0]
		ofs += variant[1]
		# Composants
		self.components = []
		for i in range (nbComponents):
			# Taille (1 à 9 octets)
			variant = variantFromBytes (data[ofs:])
			size = variant[0]
			ofs += variant[1]
			# Composant (variable)
			self.components.append (data[ofs:ofs + size])
			ofs += size
		return ofs

# https://github.com/bitcoin/bitcoin/blob/master/src/primitives/block.h
# https://en.bitcoin.it/wiki/Protocol_documentation#block
# https://en.bitcoin.it/wiki/Block_hashing_algorithm
# https://en.bitcoin.it/wiki/Difficulty
class BlockHeader:

	def validate (self, header):
		isValid = True
		if self.version != header['ver']:
			print (f'Bad version: {self.version} instead of {header["ver"]}')
			isValid = False
		previousBlock = stringToBytes (header['prev_block'])
		if self.previousBlock != previousBlock:
			print (f'Bad previous block: {self.previousBlock} instead of {previousBlock}')
			isValid = False
		merkleRoot = stringToBytes(header['mrkl_root'])
		if self.merkleTreeRoot != merkleRoot:
			print (f'Bad Merkle root: {self.merkleTreeRoot} instead of {merkleRoot}')
			isValid = False
		t = datetime.datetime.utcfromtimestamp (int (header['time']))
		if self.time != t:
			print (f'Bad time: {self.time} instead of {t}')
			isValid = False
		if self.bits != header['bits']:
			print (f'Bad bits: {self.bits} instead of {header["bits"]}')
			isValid = False
		if self.nonce != header['nonce']:
			print (f'Bad nonce: {self.nonce} instead of {header["nonce"]}')
			isValid = False
		return isValid

	def dump (self, prefix):
		s = f'{prefix}Version: {self.version}\n'
		s += f'{prefix}Previous block double SHA-256: {bytesToString (self.previousBlock)}\n'
		s += f'{prefix}Merkle tree root: {bytesToString (self.merkleTreeRoot)}\n'
		s += f'{prefix}Time: {self.time}\n'
		s += f'{prefix}Bits: {self.bits:08x} (Difficulty: {self.bits * 2 ** (8 * (0x1b - 3)):064x})\n'
		s += f'{prefix}Nonce: {self.nonce}\n'
		return s

	def serialize (self):
		data = bytearray ()
		data.extend (self.version.to_bytes (4, byteorder='little'))
		hash = bytearray (self.previousBlock)
		hash.reverse ()
		data.extend (hash)
		hash = bytearray (self.merkleTreeRoot)
		hash.reverse ()
		data.extend (hash)
		data.extend (calendar.timegm (self.time.timetuple ()).to_bytes (4, byteorder='little'))
		data.extend (self.bits.to_bytes (4, byteorder='little'))
		data.extend (self.nonce.to_bytes (4, byteorder='little'))
		return data

	def deserialize (self, data):
		ofs = 0
		# Version (4 octets)
		self.version = int.from_bytes (data[ofs:ofs + 4], byteorder='little')
		ofs += 4
		# Double SHA-256 du bloc précédent (32 octets)
		self.previousBlock = data[ofs:ofs + 32]
		self.previousBlock.reverse ()
		ofs += 32
		# Double SHA-256 racine du Merkle tree (32 octets)
		self.merkleTreeRoot = data[ofs:ofs + 32]
		self.merkleTreeRoot.reverse ()
		ofs += 32
		# Date et heure (4 octets)
		self.time = datetime.datetime.utcfromtimestamp (int.from_bytes (data[ofs:ofs + 4], byteorder='little'))
		ofs += 4
		# Difficulté sous forme compacte (4 octets)
		self.bits = int.from_bytes (data[ofs:ofs + 4], byteorder='little')
		ofs += 4
		# Nonce (4 octets) [NB : peut-être étendu à la date / heure et à la transaction coinbase]
		self.nonce = int.from_bytes (data[ofs:ofs + 4], byteorder='little')
		ofs += 4
		return ofs

# https://en.bitcoin.it/wiki/Block
class Block:

	def validate (self, block):
		# Validation de la désérialisation
		print ('*' * 80)
		isValid = self.header.validate (block)
		if len (self.transactions) != block['n_tx']:
			print (f'Bad # of transactions: {len (self.transactions)} instead of {block["n_tx"]}')
			isValid = False
		for i in range (len (self.transactions)):
			isValid = isValid and self.transactions[i].validate (block['tx'][i])
		print (f'Deserialization OK?: {isValid}')
		# Validation de la sérialisation
		isValid = True
		block = stringToBytes (block['hex'])
		data = self.serialize ()
		for i in range (len (data)):
			if data[i] != block[i]:
				print (f'Bad serialization at {i}: {data[i]} instead of {block[i]}')
				isValid = False
		print (f'Serialization OK?: {isValid}')
		print ('*' * 80)

	def dump (self, prefix=''):
		separator = f'{prefix}\t{"-" * 80}\n'
		s = f'{prefix}Header:\n'
		s += self.header.dump (prefix + '\t')
		s += f'{prefix}Transactions: {len (self.transactions)}\n'
		for i in range (len (self.transactions)):
			s += f'{separator}{prefix}\tTransaction #{i:04d}\n{separator}'
			s += self.transactions[i].dump (prefix + '\t')
		return s

	def serialize (self):
		data = bytearray ()
		data.extend (self.header.serialize ())
		data.extend (variantToBytes (len (self.transactions))[0])
		for transaction in self.transactions:
			data.extend (transaction.serialize ())
		return data

	def deserialize (self, data, progress=''):
		ofs = 0
		# Header (80 octets)
		self.header = BlockHeader ()
		ofs += self.header.deserialize (data[ofs:])
		# Nombre de transactions (1 à 9 octets)
		variant = variantFromBytes (data[ofs:])
		nbTransactions = variant[0]
		ofs += variant[1]
		# Transactions
		self.transactions = []
		for i in range (nbTransactions):
			print (f'\r{progress}Transaction: {i + 1} / {nbTransactions}', end='')
			transaction = Transaction ()
			ofs += transaction.deserialize (data[ofs:], f'\r{progress}Transaction: {i + 1} / {nbTransactions} ')
			self.transactions.append (transaction)
		return ofs

	def hashHeader (self):
		return bytearray (hashlib.sha256 (hashlib.sha256 (self.header.serialize ()).digest ()).digest ())

	def hashTransactions (self):
		hashes = []
		for transaction in self.transactions:
			hashes.append (transaction.hash ())
		return self.merkleTree ([hashes])[0][0]

	# https://en.bitcoin.it/wiki/Merkle_Trees#Merkle_Trees
	def merkleTree (self, tree):
		if len (tree[0]) == 1:
			return tree
		roots = []
		leaves = tree[0]
		if len (leaves) & 1:
			leaves.append (leaves[-1])
		for i in range (0, len (leaves), 2):
			hash = bytearray (leaves[i])
			hash.extend (leaves[i + 1])
			roots.append (bytearray (hashlib.sha256 (hashlib.sha256 (hash).digest ()).digest ()))
		tree.insert (0, roots)
		return self.merkleTree (tree)

path = 'data/'
# Le premier bloc contenant une transaction dont une input fait référence à une output d'une transaction antérieure
file170 = '000000-000170'
# Le dernier bloc avant SegWit (1696 transactions) et le suivant (1866 transactions)
file481823 = '481823-481824'
# Le premier bloc avec SegWit (1866 transactions)
file481824 = '481824'
# Le dernier bloc en date au moment où j'écris ce code (2489 transactions)
file709703 = '709703'

filename = file709703
file = open (f'{path}{filename}.json', 'rt')
data = file.read ()
file.close ()
data = json.loads (data)
blocks = []
dump = ''
separator = '*' * 80 + '\n'
for i in range (len (data)):
	print ('-' * 80)
	# Désérialisation à partir de l'entrée "hex" du JSON du bloc
	print (f'Block: {i + 1} / {len (data)}', end='')
	block = Block ()
	blocks.append (block)
	block.deserialize (stringToBytes (data[i]['hex']), f'Block: {i + 1} / {len (data)} ')
	print ('')
	# Vérification de la désérialisation avec le JSON
	# block.validate (data[i])
	# Calcul du hash de la racine du Merkle tree des transactions et vérification avec le JSON
	isHashOK = True
	hash = block.hashTransactions ()
	hash.reverse ()  # Il faudra qu'on m'explique...
	if block.header.merkleTreeRoot != hash:
		isHashOK = False
	print (f'Transactions hash OK?: {isHashOK}')
	# Calcul du hash du header bloc précédent et vérification avec le JSON
	if i:
		isHashOK = True
		hash = blocks[i - 1].hashHeader ()
		hash.reverse ()		# Il faudra qu'on m'explique...
		if blocks[i].header.previousBlock != hash:
			isHashOK = False
		print (f'Previous block hash OK?: {isHashOK}')
	# Dump de la désérialisation sous forme de texte
	dump += f'{separator}Block #{i:06d}\n{separator}\n{block.dump ()}\n'
	# Dump de la sérialisation sous forme binaire
	file = open (f'{path}blocks/{int (filename[:6]) + i:06d}.bin', 'wb')
	file.write (block.serialize ())
	file.close ()
print ('-' * 80)
print (f'Dumping to "{path}{filename}.txt"')
file = open (f'{path}{filename}.txt', 'wt')
file.write (dump)
file.close ()
print ('Done')