# Réalisé par Denis Duplan pour Stash of Code (https://www.stashofcode.fr) en novembre 2021.

# Cette oeuvre est mise à disposition selon les termes de la Licence Creative Commons Attribution - Pas d’Utilisation Commerciale 4.0 International

def bytesToString (data: bytearray) -> str:
	return ''.join ([f'{b:02x}' for b in data])

def bytesToASCII (data: bytearray) -> str:
	return ''.join ([chr (b) if b in range (0x20, 0x7f) else '.' for b in data])

def stringToBytes (data: str) -> bytearray:
	return bytearray ([int (data[i:i + 2], 16) for i in range (0, len (data), 2)])

# https://en.bitcoin.it/wiki/Protocol_documentation#Variable_length_integer
def variantToBytes (n: int) -> (bytearray, int):
	data = bytearray ()
	if n < 0xfd:
		size = 1
	elif n < 0xffff:
		size = 2
		data.append (0xfd)
	elif n < 0xffffffff:
		size = 4
		data.append (0xfe)
	else:
		size = 8
		data.append (0xff)
	data.extend (n.to_bytes (size, byteorder='little'))
	return data, len (data)

def variantFromBytes (data: bytearray) -> (int, int):
	if data[0] == 0xff:
		ofs = 1
		size = 8
	elif data[0] == 0xfe:
		ofs = 1
		size = 4
	elif data[0] == 0xfd:
		ofs = 1
		size = 2
	else:
		ofs = 0
		size = 1
	return int.from_bytes (data[ofs:ofs + size], byteorder='little'), ofs + size

# https://en.bitcoin.it/wiki/Script
# https://github.com/bitcoin/bitcoin/blob/master/src/script/script.h
# La pile contient des "bytes vectors", de séquences de 520 octets au plus. Quand une séquence est un nombre, c'est un entier sur 4 octets en little endian, dont le bit le plus significatif donne le signe (ex : 0x81 = -1). Noter que l'addition et la soustraction peuvent produire un entier sur 5 octets. Quand c'est un booléen, c'est la valeur False si au moins un des octets n'est pas nul.
def decodeScript (script, prefix=''):
	opcodes = {
		0x61: 'OP_NOP',
		0x63: 'OP_IF',
		0x64: 'OP_NOTIF',
		0x67: 'OP_ELSE',
		0x68: 'OP_ENDIF',
		0x69: 'OP_VERIFY',
		0x6a: 'OP_RETURN',
		0x6b: 'OP_TOALTSTACK',
		0x6c: 'OP_FROMALTSTACK',
		0x73: 'OP_IFDUP',
		0x74: 'OP_DEPTH',
		0x75: 'OP_DROP',
		0x76: 'OP_DUP',
		0x77: 'OP_NIP',
		0x78: 'OP_OVER',
		0x79: 'OP_PICK',
		0x7a: 'OP_ROLL',
		0x7b: 'OP_ROT',
		0x7c: 'OP_SWAP',
		0x7d: 'OP_TUCK',
		0x6d: 'OP_2DROP',
		0x6e: 'OP_2DUP',
		0x6f: 'OP_3DUP',
		0x70: 'OP_2OVER',
		0x71: 'OP_2ROT',
		0x72: 'OP_2SWAP',
		0x7e: 'OP_CAT',
		0x7f: 'OP_SUBSTR',
		0x80: 'OP_LEFT',
		0x81: 'OP_RIGHT',
		0x82: 'OP_SIZE',
		0x83: 'OP_INVERT',
		0x84: 'OP_AND',
		0x85: 'OP_OR',
		0x86: 'OP_XOR',
		0x87: 'OP_EQUAL',
		0x88: 'OP_EQUALVERIFY',
		0x8b: 'OP_1ADD',
		0x8c: 'OP_1SUB',
		0x8d: 'OP_2MUL',
		0x8e: 'OP_2DIV',
		0x8f: 'OP_NEGATE',
		0x90: 'OP_ABS',
		0x91: 'OP_NOT',
		0x92: 'OP_0NOTEQUAL',
		0x93: 'OP_ADD',
		0x94: 'OP_SUB',
		0x95: 'OP_MUL',
		0x96: 'OP_DIV',
		0x97: 'OP_MOD',
		0x98: 'OP_LSHIFT',
		0x99: 'OP_RSHIFT',
		0x9a: 'OP_BOOLAND',
		0x9b: 'OP_BOOLOR',
		0x9c: 'OP_NUMEQUAL',
		0x9d: 'OP_NUMEQUALVERIFY',
		0x9e: 'OP_NUMNOTEQUAL',
		0x9f: 'OP_LESSTHAN',
		0xa0: 'OP_GREATERTHAN',
		0xa1: 'OP_LESSTHANOREQUAL',
		0xa2: 'OP_GREATERTHANOREQUAL',
		0xa3: 'OP_MIN',
		0xa4: 'OP_MAX',
		0xa5: 'OP_WITHIN',
		0xa6: 'OP_RIPEMD160',
		0xa7: 'OP_SHA1',
		0xa8: 'OP_SHA256',
		0xa9: 'OP_HASH160',
		0xaa: 'OP_HASH256',
		0xab: 'OP_CODESEPARATOR',
		0xac: 'OP_CHECKSIG',
		0xad: 'OP_CHECKSIGVERIFY',
		0xae: 'OP_CHECKMULTISIG',
		0xaf: 'OP_CHECKMULTISIGVERIFY',
		0xb1: 'OP_CHECKLOCKTIMEVERIFY (ex-OP_NOP2)',
		0xb2: 'OP_CHECKSEQUENCEVERIFY (ex-OP_NOP3)',
		0xfd: 'OP_PUBKEYHASH',
		0xfe: 'OP_PUBKEY',
		0xff: 'OP_INVALIDOPCODE',
		0x50: 'OP_RESERVED',
		0x62: 'OP_VER',
		0x65: 'OP_VERIF',
		0x66: 'OP_VERNOTIF',
		0x89: 'OP_RESERVED1',
		0x8a: 'OP_RESERVED2',
		0xb0: 'OP_NOP1',
		0xb3: 'OP_NOP4',
		0xb4: 'OP_NOP5',
		0xb5: 'OP_NOP6',
		0xb6: 'OP_NOP7',
		0xb7: 'OP_NOP8',
		0xb8: 'OP_NOP9',
		0xb9: 'OP_NOP10'
	}
	s = ''
	i = 0
	while i < len (script):
		b = script[i]
		if b == 0:
			s += f'{prefix}PUSH empty array\n'
			i += 1
		elif b >= 1 and b <= 75:
			i += 1
			s += f'{prefix}PUSH {b} bytes: {bytesToString (script[i:i + b])}\n'
			s += prefix + len (f'PUSH {b} bytes: ') * ' ' + f'{bytesToASCII (script[i:i + b])}\n'
			i += b
		elif b >= 76 and b <= 78:
			b = int.from_bytes (script[i:i + 76 - b + 1], byteorder='little')
			i += 76 - b + 1
			s += f'{prefix}PUSH {b} bytes: {bytesToString (script[i:i + b])}\n'
			s += prefix + len (f'PUSH {b} bytes: ') * ' ' + f'{bytesToASCII (script[i:i + b])}\n'
			i += b
		elif b == 79:
			s += f'{prefix}PUSH -1\n'
			i += 1
		elif b == 81:
			s += f'{prefix}PUSH 1\n'
			i += 1
		elif b >= 82 and b <= 96:
			s += f'{prefix}PUSH {b - 82 + 2}\n'
			i += 1
		else:
			if b not in opcodes:
				s += f'{prefix}*UNKNOWN*: {bytesToString (script[i:])}\n'
				s += prefix + len (f'*UNKNOWN*: ') * ' ' + f'{bytesToASCII (script[i:])}\n'
				break
			else:
				s += f'{prefix}{opcodes[b]}\n'
				i += 1
	return s