# Réalisé par Denis Duplan pour Stash of Code (https://www.stashofcode.fr) en novembre 2021.

# Cette oeuvre est mise à disposition selon les termes de la Licence Creative Commons Attribution - Pas d’Utilisation Commerciale 4.0 International

import datetime
import calendar
import requests
import hashlib
from btc.tools import *

# https://github.com/bitcoin/bitcoin/blob/master/src/primitives/transaction.h
# https://en.bitcoin.it/wiki/Protocol_documentation#tx
# https://en.bitcoin.it/wiki/Transactions
class Transaction:

	def __init__ (self):
		self.version: int = 0
		self.inputs: [TransactionInput] = []
		self.outputs: [TransactionOutput] = []
		self.witnesses: [Witness] = []
		self.lockTime: int = 0

	def serialize (self, withWitnesses: bool = True) -> bytearray:
		data = bytearray ()
		data.extend (self.version.to_bytes (4, 'little'))
		if withWitnesses:
			if len (self.witnesses):
				data.append (0x00)
				data.append (0x01)
		data.extend (variantToBytes (len (self.inputs))[0])
		for txIn in self.inputs:
			data.extend (txIn.serialize ())
		data.extend (variantToBytes (len (self.outputs))[0])
		for txOut in self.outputs:
			data.extend (txOut.serialize ())
		if withWitnesses:
			if len (self.witnesses):
				for witness in self.witnesses:
					data.extend (witness.serialize ())
		data.extend (self.lockTime.to_bytes (4, 'little'))
		return data

	def deserialize (self, data: bytearray, progress: str = '') -> int:
		ofs = 0
		# Version (4 octets)
		self.version = int.from_bytes (data[ofs:ofs + 4], 'little')
		ofs += 4
		# Présence de témoins (4 octets, si 0x0001)
		hasWitnesses = False
		if int.from_bytes (data[ofs:ofs + 2], 'little') == 0x0100:
			hasWitnesses = True
			ofs += 2
		# Nombre d'entrées (1 à 9 octets)
		variant = variantFromBytes (data[ofs:])
		nbInputs = variant[0]
		ofs += variant[1]
		# Entrées
		self.inputs = []
		for i in range (nbInputs):
			print (f'\r{progress}Input: {i + 1} / {nbInputs}', end='')
			transactionInput = TransactionInput ()
			ofs += transactionInput.deserialize (data[ofs:])
			self.inputs.append (transactionInput)
		# Nombre de sorties (1 à 9 octets)
		variant = variantFromBytes (data[ofs:])
		nbOutputs = variant[0]
		ofs += variant[1]
		# Sorties
		self.outputs = []
		for i in range (nbOutputs):
			print (f'\r{progress}Output: {i + 1} / {nbOutputs}', end='')
			transactionOutput = TransactionOutput ()
			ofs += transactionOutput.deserialize (data[ofs:])
			self.outputs.append (transactionOutput)
		# Témoins
		self.witnesses = []
		if hasWitnesses:
			for i in range (nbInputs):
				print (f'\r{progress}Witness: {i + 1} / {nbInputs}', end='')
				witness = Witness ()
				ofs += witness.deserialize (data[ofs:])
				self.witnesses.append (witness)
		# Hauteur du bloc ou date & heure de finalisation du bloc (4 octets)
		self.lockTime = int.from_bytes (data[ofs:ofs + 4], 'little')
		ofs += 4
		return ofs

	def dump (self, prefix: str) -> str:
		separator = f'{prefix}\t{"-" * 80}\n'
		s = f'{prefix}Version: {self.version}\n'
		if self.lockTime == 0:
			lockTime = 'Not locked'
		elif self.lockTime < 500000000:
			lockTime = f'Block #{self.lockTime}'
		else:
			lockTime = datetime.datetime.utcfromtimestamp (self.lockTime)
		s += f'{prefix}Lock time: {lockTime}\n'
		s += f'{prefix}Inputs: {len (self.inputs)}\n'
		for i in range (len (self.inputs)):
			s += f'{separator}{prefix}\tInput #{i:03d}\n{separator}'
			s += self.inputs[i].dump (prefix + '\t')
		s += f'{prefix}Outputs: {len (self.outputs)}\n'
		for i in range (len (self.outputs)):
			s += f'{separator}{prefix}\tOutput #{i:03d}\n{separator}'
			s += self.outputs[i].dump (prefix + '\t')
		s += f'{prefix}Witnesses: {len (self.witnesses)}\n'
		for i in range (len (self.witnesses)):
			s += f'{separator}{prefix}\tWitness #{i:03d}\n{separator}'
			s += self.witnesses[i].dump (prefix + '\t')
		return s

	# https://github.com/bitcoin/bips/blob/master/bip-0141.mediawiki
	def hash (self) -> bytearray:
		return bytearray (hashlib.sha256 (hashlib.sha256 (self.serialize (withWitnesses=False)).digest ()).digest ())

# https://en.bitcoin.it/wiki/Protocol_documentation#tx
# https://en.bitcoin.it/wiki/Transactions
class TransactionInput:

	def __init__ (self):
		self.transaction: bytearray = bytearray ()
		self.output: int = 0
		self.scriptSig: bytearray = bytearray ()
		self.sequence: int = 0

	def serialize (self) -> bytearray:
		data = bytearray ()
		hash = bytearray (self.transaction)
		hash.reverse ()
		data.extend (hash)
		data.extend (self.output.to_bytes (4, 'little'))
		data.extend (variantToBytes (len (self.scriptSig))[0])
		data.extend (self.scriptSig)
		data.extend (self.sequence.to_bytes (4, 'little'))
		return data

	def deserialize (self, data: bytearray) -> int:
		ofs = 0
		# Double SHA-256 de la transaction précédente (32 octets)
		self.transaction = data[ofs:ofs + 32]
		self.transaction.reverse ()
		ofs += 32
		# Indice de la sortie de la transaction précédente (4 octets)
		self.output = int.from_bytes (data[ofs:ofs + 4], 'little')
		ofs += 4
		# Taille du script (1 à 9 octets)
		variant = variantFromBytes (data[ofs:])
		scriptSigSize = variant[0]
		ofs += variant[1]
		# Script
		self.scriptSig = data[ofs:ofs + scriptSigSize]
		ofs += scriptSigSize
		# Séquence (4 octets)
		self.sequence = int.from_bytes (data[ofs:ofs + 4], 'little')
		ofs += 4
		return ofs

	def dump (self, prefix: str) -> str:
		s = f'{prefix}Previous transaction double SHA-256: {bytesToString (self.transaction)}\n'
		s += f'{prefix}Previous transaction output: {self.output}\n'
		s += f'{prefix}ScriptSig: {bytesToString (self.scriptSig)}\n'
		s += decodeScript (self.scriptSig, f'{prefix}\t')
		s += f'{prefix}Sequence: {self.sequence}\n'
		return s

# https://en.bitcoin.it/wiki/Protocol_documentation#tx
# https://en.bitcoin.it/wiki/Transactions
class TransactionOutput:

	def __init__ (self):
		self.value: int = 0
		self.scriptPubKey: bytearray = bytearray ()

	def serialize (self) -> bytearray:
		data = bytearray ()
		data.extend (self.value.to_bytes (8, 'little'))
		data.extend (variantToBytes (len (self.scriptPubKey))[0])
		data.extend (self.scriptPubKey)
		return data

	def deserialize (self, data: bytearray) -> int:
		ofs = 0
		# Valeur en Satoshis (ie : BTC / 10^8) (8 octets)
		self.value = int.from_bytes (data[ofs:ofs + 8], 'little')
		ofs += 8
		# Taille du script (1 à 9 octets)
		variant = variantFromBytes (data[ofs:])
		scriptPubKeySize = variant[0]
		ofs += variant[1]
		# Script
		self.scriptPubKey = data[ofs:ofs + scriptPubKeySize]
		ofs += scriptPubKeySize
		return ofs

	def dump (self, prefix: str) -> str:
		s = f'{prefix}Value: {self.value} Satoshis ({self.value / 10**8} BTC)\n'
		s += f'{prefix}ScriptPubKey: {bytesToString (self.scriptPubKey)}\n'
		s += decodeScript (self.scriptPubKey, f'{prefix}\t')
		return s

# https://en.bitcoin.it/wiki/Protocol_documentation#tx
# https://en.bitcoin.it/wiki/Segregated_Witness
class Witness:

	def __init__ (self):
		self.components: [bytearray] = []

	def serialize (self) -> bytearray:
		data = bytearray ()
		data.extend (variantToBytes (len (self.components))[0])
		for component in self.components:
			data.extend (variantToBytes (len (component))[0])
			data.extend (component)
		return data

	def deserialize (self, data: bytearray) -> int:
		ofs = 0
		# Nombre de composants (1 à 9 octets)
		variant = variantFromBytes (data[ofs:])
		nbComponents = variant[0]
		ofs += variant[1]
		# Composants
		self.components = []
		for i in range (nbComponents):
			# Taille (1 à 9 octets)
			variant = variantFromBytes (data[ofs:])
			size = variant[0]
			ofs += variant[1]
			# Composant (variable)
			self.components.append (data[ofs:ofs + size])
			ofs += size
		return ofs

	def dump (self, prefix: str) -> str:
		separator = f'{prefix}\t{"-" * 80}\n'
		s = f'{prefix}# of components: {len (self.components)}\n'
		s += f'{prefix}Components: {len (self.components)}\n'
		for i in range (len (self.components)):
			s += f'{separator}{prefix}\tComponent #{i:03d}\n{separator}'
			s += f'{prefix}\tSize: {len (self.components[i])}\n'
			s += f'{prefix}\tData: {bytesToString (self.components[i])}\n'
		return s

# https://github.com/bitcoin/bitcoin/blob/master/src/primitives/block.h
# https://en.bitcoin.it/wiki/Protocol_documentation#block
# https://en.bitcoin.it/wiki/Block_hashing_algorithm
# https://en.bitcoin.it/wiki/Difficulty
class BlockHeader:

	def __init__ (self):
		self.version: int = 0
		self.previousBlock: bytearray = bytearray ()
		self.merkleTreeRoot: bytearray = bytearray ()
		self.time: datetime.datetime = datetime.datetime.now ()
		self.bits: int = 0
		self.nonce: int = 0

	def serialize (self) -> bytearray:
		data = bytearray ()
		data.extend (self.version.to_bytes (4, 'little'))
		hash = bytearray (self.previousBlock)
		hash.reverse ()
		data.extend (hash)
		hash = bytearray (self.merkleTreeRoot)
		hash.reverse ()
		data.extend (hash)
		data.extend (calendar.timegm (self.time.timetuple ()).to_bytes (4, 'little'))
		data.extend (self.bits.to_bytes (4, 'little'))
		data.extend (self.nonce.to_bytes (4, 'little'))
		return data

	def deserialize (self, data: bytearray) -> int:
		ofs = 0
		# Version (4 octets)
		self.version = int.from_bytes (data[ofs:ofs + 4], 'little')
		ofs += 4
		# Double SHA-256 du bloc précédent (32 octets)
		self.previousBlock = data[ofs:ofs + 32]
		self.previousBlock.reverse ()
		ofs += 32
		# Double SHA-256 racine du Merkle tree (32 octets)
		self.merkleTreeRoot = data[ofs:ofs + 32]
		self.merkleTreeRoot.reverse ()
		ofs += 32
		# Date et heure (4 octets)
		self.time = datetime.datetime.utcfromtimestamp (int.from_bytes (data[ofs:ofs + 4], 'little'))
		ofs += 4
		# Difficulté sous forme compacte (4 octets)
		self.bits = int.from_bytes (data[ofs:ofs + 4], 'little')
		ofs += 4
		# Nonce (4 octets) [NB : peut-être étendu à la date / heure et à la transaction coinbase]
		self.nonce = int.from_bytes (data[ofs:ofs + 4], 'little')
		ofs += 4
		return ofs

	def dump (self, prefix: str) -> str:
		s = f'{prefix}Version: {self.version}\n'
		s += f'{prefix}Previous block double SHA-256: {bytesToString (self.previousBlock)}\n'
		s += f'{prefix}Merkle tree root: {bytesToString (self.merkleTreeRoot)}\n'
		s += f'{prefix}Time: {self.time}\n'
		s += f'{prefix}Bits: {self.bits:08x} (Difficulty: {self.bits * 2 ** (8 * (0x1b - 3)):064x})\n'
		s += f'{prefix}Nonce: {self.nonce}\n'
		return s

# https://en.bitcoin.it/wiki/Block
class Block:

	def __init__ (self):
		self.header: BlockHeader = BlockHeader ()
		self.transactions: [Transaction] = []

	def serialize (self) -> bytearray:
		data = bytearray ()
		data.extend (self.header.serialize ())
		data.extend (variantToBytes (len (self.transactions))[0])
		for transaction in self.transactions:
			data.extend (transaction.serialize ())
		return data

	def deserialize (self, data: bytearray, progress: str = '') -> int:
		ofs = 0
		# Header (80 octets)
		self.header = BlockHeader ()
		ofs += self.header.deserialize (data[ofs:])
		# Nombre de transactions (1 à 9 octets)
		variant = variantFromBytes (data[ofs:])
		nbTransactions = variant[0]
		ofs += variant[1]
		# Transactions
		self.transactions = []
		for i in range (nbTransactions):
			print (f'\r{progress}Transaction: {i + 1} / {nbTransactions}', end='')
			transaction = Transaction ()
			ofs += transaction.deserialize (data[ofs:], f'\r{progress}Transaction: {i + 1} / {nbTransactions} ')
			self.transactions.append (transaction)
		return ofs

	def dump (self, prefix: str = '') -> str:
		separator = f'{prefix}\t{"-" * 80}\n'
		s = f'{prefix}Header:\n'
		s += self.header.dump (prefix + '\t')
		s += f'{prefix}Transactions: {len (self.transactions)}\n'
		for i in range (len (self.transactions)):
			s += f'{separator}{prefix}\tTransaction #{i:04d}\n{separator}'
			s += self.transactions[i].dump (prefix + '\t')
		return s

	def hashHeader (self) -> bytearray:
		return bytearray (hashlib.sha256 (hashlib.sha256 (self.header.serialize ()).digest ()).digest ())

	def hashTransactions (self) -> bytearray:
		hashes = []
		for transaction in self.transactions:
			hashes.append (transaction.hash ())
		return self.merkleTree ([hashes])[0][0]

	# https://en.bitcoin.it/wiki/Merkle_Trees#Merkle_Trees
	def merkleTree (self, tree: []) -> []:
		if len (tree[0]) == 1:
			return tree
		roots = []
		leaves = tree[0]
		if len (leaves) & 1:
			leaves.append (leaves[-1])
		for i in range (0, len (leaves), 2):
			hash = bytearray (leaves[i])
			hash.extend (leaves[i + 1])
			roots.append (bytearray (hashlib.sha256 (hashlib.sha256 (hash).digest ()).digest ()))
		tree.insert (0, roots)
		return self.merkleTree (tree)

# De quoi simuler la détention de l'intégralité de la blockchain en chargeant le binaire des blocs depuis le même service que celui utilisé dans downloadBlocks.py
class Blockchain:
	def __init__ (self):
		self.blocks: [Block] = []

	def getBlockByHash (self, hash: bytearray) -> Block:
		for block in self.blocks:
			if block.hashHeader () == hash:
				return block
		print (f'Block {bytesToString (hash)} not found in downloaded blocks. Downloading it...')
		url = 'https://blockchain.info/rawblock/'
		response = requests.get (f'{url}{bytesToString (hash)}?format=hex')
		if response.status_code != 200:
			print (f'HTTP Error {response.status_code} while downloading block {bytesToString (hash)}')
			return None
		block = Block ()
		block.deserialize (stringToBytes (response.content.decode ('utf-8')))
		self.blocks.append (block)
		print (f'\nBlock {bytesToString (hash)} deserialized')
		return block

	def getTransactionByHash (self, hash: bytearray) -> Transaction:
		for block in self.blocks:
			for transaction in block.transactions:
				if transaction.hash () == hash[::-1]:
					return transaction
		print (f'Transaction {bytesToString (hash)} not found in downloaded blocks')
		return None
